﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using MicroRWD.Common;

namespace MicroRWD.QT
{
    // Quad tag reader
    public class QuadTagReader : Reader
    {
        #region Private Constants

        // The read tag command (ASCII 'R')
        private byte C_CMD_TAG_READ = 0x52;

        // The write tag command (ASCII 'W')
        private byte C_CMD_TAG_WRITE = 0x57;

        // The reader type command (ASCII 'v')
        private byte C_CMD_READER_TYPE = 0x76;

        #endregion


        #region Private Attributes

        // Reader status changed event handler
        private event StatusUpdateEventHandler statusUpdateEvent;

        #endregion


        #region Public Properties

        // Mode
        public Mode Mode { get; private set; }

        // Status update event handler
        public event StatusUpdateEventHandler StatusUpdateEvent { add { statusUpdateEvent += value; } remove { statusUpdateEvent -= value; } }

        #endregion


        #region Constructor

        // Constructs a new Quad Tag reader instance attached via the specified port name
        public QuadTagReader(string _portName, int _commandTimeoutMillis) : base(_portName, _commandTimeoutMillis)
        {
            // Initialise mode to none
            Mode = Mode.NONE;

            // Register for acknowledge events
            base.AcknowledgeEvent += MyAcknowledgeEventHandler;
        }

        #endregion

        
        #region Public Methods

        public new bool CmdProgram(byte location, byte value)
        {
            bool result = false;

            // Delegate to generic reader
            byte ack = base.CmdProgram(location, value);

            // Check for success (got response with clear status flags)
            if ((ack & 0xD1) == 0xC0)
            {
                result = true;
            }

            return result;
        }

        // Select reader type
        public bool CmdReaderType(ReaderType _type)
        {
            bool result = false;

            // Delegate to base class
            byte[] reply = CmdGeneric(new byte[] { C_CMD_READER_TYPE, (byte)_type });

            // If we get a response check acknowledge byte
            if (reply.Length == 1)
            {
                byte ack = reply[0];
                if ((ack & 0xD1) == 0xC0)
                {
                    result = true;
                }
            }

            // Return result
            return result;
        }

        // Tag Read (HITAG1/S, HITAG2, EM400X, MC200)
        public byte[] CmdTagRead(byte _page)
        {
            // Zero length reply on timeout, data otherwise
            byte[] result = new byte[0];

            // Check mode and parameters
            if (((Mode == Mode.HITAG1_S) && (_page >= 0) && (_page <= 0x3f)) ||
               ((Mode == Mode.HITAG2) && (_page >= 0) && (_page <= 0x07)) ||
               ((Mode == Mode.EM400X) && (_page == 0)) ||
               ((Mode == Mode.MC200) && (_page == 0)))
            {
                // Delegate to base class
                result = CmdGeneric(new byte[] { C_CMD_TAG_READ, _page });
            }
            else
            {
                // Error
                Log.Error("invalid parameters");
            }

            // Return result
            return result;
        }

        // Tag Write (HITAG1/S, HITAG2)
        public byte[] CmdTagWrite(byte _page, byte[] _data)
        {
            // Zero length reply on timeout, data otherwise
            byte[] result = new byte[0];

            // Check mode and parameters
            if (((Mode == Mode.HITAG1_S) && (_page >= 0) && (_page <=0x3f) && (_data != null) && (_data.Length == 4)) ||
                ((Mode == Mode.HITAG2) && (_page >= 0) && (_page <= 0x07) && (_data != null) && (_data.Length == 4)))
            {
                // Delegate to base class
                result = CmdGeneric(new byte[] { C_CMD_TAG_WRITE, _page, _data[0], _data[1], _data[2], _data[3] });
            }
            else
            {
                // Error
                Log.Error("invalid parameters");
            }

            // Return result
            return result;
        }

        // Tag Status
        public new Status CmdTagStatus()
        {
            Status result = new Status();

            // Delegate to generic reader
            byte ack = base.CmdStatus();

            // Check for success (got response)
            if ((ack & 0xC0) == 0xC0)
            {
                result = new Status(ack);
            }

            return result;
        }

        // Set mode
        public bool SetMode(Mode _mode)
        {
            bool result = false;

            switch (_mode)
            {
                case Mode.HITAG1_S:
                    if (CmdReaderType(ReaderType.Hitag1_S))
                    {
                        Mode = Mode.HITAG1_S;
                        result = true;
                    }
                    break;

                case Mode.HITAG2:
                    if (CmdReaderType(ReaderType.Hitag2))
                    {
                        Mode = Mode.HITAG2;
                        result = true;
                    }
                    break;

                case Mode.EM400X:
                    if (CmdReaderType(ReaderType.EM400X_MC200))
                    {
                        if (CmdProgram(0x10, 0x01))
                        {
                            Mode = Mode.EM400X;
                            result = true;
                        }
                    }
                    break;

                case Mode.MC200:
                    if (CmdReaderType(ReaderType.EM400X_MC200))
                    {
                        if (CmdProgram(0x10, 0x00))
                        {
                            Mode = Mode.MC200;
                            result = true;
                        }
                    }
                    break;
            }

            return result;
        }

        #endregion

        
        #region Private Methods

        // Fire status update event to listeners
        private void FireStatusUpdateEvent(Status _status)
        {
            try
            {
                // Instantiate suitable event args object
                StatusUpdateEventArgs args = new StatusUpdateEventArgs(_status);

                // Check for current listeners
                if (statusUpdateEvent != null)
                {
                    // Dispatch event to listeners
                    statusUpdateEvent(this, args);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        // Handle acknowledge events from base reader
        private void MyAcknowledgeEventHandler(object sender, AcknowledgeEventArgs args)
        {
            // Decode byte into Status instance
            Status status = new Status(args.Value);

            // Notify listeners using a Status instance
            FireStatusUpdateEvent(status);
        }

        #endregion

    }
}
